<?php
/**
 * Plugin Name: Chat Relay
 * Description: A minimalist, editorial-style WhatsApp bridge for modern brands.
 * Version: 7.2.0
 * Author: Edgely
 * Author URI: https://relay.edgely.link
 * Text Domain: chat-relay
 */

if (!defined('ABSPATH')) exit; 

define('CHAT_RELAY_VERSION', '7.2.0');
define('CHAT_RELAY_URL', plugin_dir_url(__FILE__));
define('CHAT_RELAY_PATH', plugin_dir_path(__FILE__));

// 1. MENU
add_action('admin_menu', 'chat_relay_create_menu');
function chat_relay_create_menu() {
    add_menu_page('Chat Relay', 'Chat Relay', 'manage_options', 'chat-relay', 'chat_relay_settings_page', 'dashicons-whatsapp', 100);
}

add_action('admin_head', 'chat_relay_admin_icon_style');
function chat_relay_admin_icon_style() {
    ?>
    <style>
        #toplevel_page_chat-relay .wp-menu-image:before { display: none; }
        #toplevel_page_chat-relay .wp-menu-image {
            background-color: #a7aaad;
            -webkit-mask-image: url('data:image/svg+xml;base64,PHN2ZyB2aWV3Qm94PSIwIDAgMjQgMjQiIHhtbG5zPSJodHRwOi8vd3d3LnczLm9yZy8yMDAwL3N2ZyI+PHBhdGggZD0iTTEyIDJMMTQuNSA5LjVMMjIgMTJMMTQuNSAxNC41TDEyIDIyTDkuNSAxNC41TDIgMTJMOS41IDkuNUwxMiAyWiIvPjwvc3ZnPg==');
            mask-image: url('data:image/svg+xml;base64,PHN2ZyB2aWV3Qm94PSIwIDAgMjQgMjQiIHhtbG5zPSJodHRwOi8vd3d3LnczLm9yZy8yMDAwL3N2ZyI+PHBhdGggZD0iTTEyIDJMMTQuNSA5LjVMMjIgMTJMMTQuNSAxNC41TDEyIDIyTDkuNSAxNC41TDIgMTJMOS41IDkuNUwxMiAyWiIvPjwvc3ZnPg==');
            -webkit-mask-repeat: no-repeat; mask-repeat: no-repeat;
            -webkit-mask-position: center; mask-position: center;
            -webkit-mask-size: 20px; mask-size: 20px;
            transition: background-color 0.1s;
        }
        #toplevel_page_chat-relay:hover .wp-menu-image,
        #toplevel_page_chat-relay.wp-has-current-submenu .wp-menu-image { background-color: #fff !important; }
    </style>
    <?php
}

// 2. SETTINGS
add_action('admin_init', 'chat_relay_register_settings');
function chat_relay_register_settings() {
    register_setting('chat_relay_group', 'chat_relay_phone', 'sanitize_text_field');
    register_setting('chat_relay_group', 'chat_relay_brand', 'sanitize_text_field');
    register_setting('chat_relay_group', 'chat_relay_color', 'sanitize_hex_color');
    register_setting('chat_relay_group', 'chat_relay_theme', 'sanitize_text_field');
    register_setting('chat_relay_group', 'chat_relay_font_mode', 'sanitize_text_field');
    register_setting('chat_relay_group', 'chat_relay_welcome', 'sanitize_text_field');
    register_setting('chat_relay_group', 'chat_relay_question', 'sanitize_text_field');
    register_setting('chat_relay_group', 'chat_relay_privacy_url', 'esc_url_raw');
}

// 3. ASSETS & INJECTION (UPDATED FONT)
add_action('wp_enqueue_scripts', 'chat_relay_enqueue_assets');
function chat_relay_enqueue_assets() {
    $phone = get_option('chat_relay_phone');
    if(empty($phone)) return;

    //DEFAULT FONT: Inter
    if (get_option('chat_relay_font_mode', 'plugin') === 'plugin') {
        wp_enqueue_style('chat-relay-font', 'https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600&display=swap', [], null);
    }

    wp_enqueue_style('chat-relay-css', CHAT_RELAY_URL . 'assets/style.css', [], CHAT_RELAY_VERSION);

    $color = get_option('chat_relay_color', '#111111');
    $theme = get_option('chat_relay_theme', 'light');
    $font_mode = get_option('chat_relay_font_mode', 'plugin');
    
    //DEFAULT FONT: Inter
    if ($font_mode === 'inherit') {
        $font_stack = 'inherit'; 
    } else {
        $font_stack = '"Inter", -apple-system, BlinkMacSystemFont, Roboto, sans-serif';
    }
    
    // SCOPED VARIABLES
    if ($theme === 'dark') {
        $css_vars = "
        #chat-relay-widget-container { 
            --chat-relay-brand: {$color}; 
            --chat-relay-bg: #121212; 
            --chat-relay-surface: #1E1E1E; 
            --chat-relay-bubble: #2C2C2C; 
            --chat-relay-text: #E0E0E0; 
            --chat-relay-sub: #A0A0A0; 
            --chat-relay-border: #333333; 
            --chat-relay-input: #1E1E1E; 
            --chat-relay-backdrop: rgba(0,0,0,0.7); 
            --chat-relay-avatar: #ffffff;
            --chat-relay-hover: rgba(255,255,255,0.1); 
            --chat-relay-font: {$font_stack}; 
        }";
    } else {
        $css_vars = "
        #chat-relay-widget-container { 
            --chat-relay-brand: {$color}; 
            --chat-relay-bg: #ffffff; 
            --chat-relay-surface: #ffffff; 
            --chat-relay-bubble: #f3f3f3; 
            --chat-relay-text: #111111; 
            --chat-relay-sub: #555555; 
            --chat-relay-border: #eaeaea; 
            --chat-relay-input: #ffffff; 
            --chat-relay-backdrop: rgba(0,0,0,0.3); 
            --chat-relay-avatar: #111111;
            --chat-relay-hover: rgba(0,0,0,0.05); 
            --chat-relay-font: {$font_stack}; 
        }";
    }
    wp_add_inline_style('chat-relay-css', $css_vars);

    wp_enqueue_script('chat-relay-js', CHAT_RELAY_URL . 'assets/script.js', [], CHAT_RELAY_VERSION, true);

    global $post;
    $current_title = is_singular() ? get_the_title() : 'General Inquiry';
    
    wp_localize_script('chat-relay-js', 'chatRelayData', [
        'phone' => $phone,
        'brand' => get_option('chat_relay_brand', 'The Studio'),
        'welcome' => get_option('chat_relay_welcome', 'Welcome.'),
        'question' => get_option('chat_relay_question', "How can we help you today?"),
        'pageTitle' => $current_title
    ]);
}

// 4. HTML OUTPUT
add_action('wp_footer', 'chat_relay_render_html');
function chat_relay_render_html() {
    if(empty(get_option('chat_relay_phone'))) return;
    $privacy_url = get_option('chat_relay_privacy_url');
    ?>
    <div id="chat-relay-widget-container" aria-hidden="false">
        <button class="chat-relay-trigger" id="chat-relay-trigger-btn" aria-label="Open Chat" onclick="toggleChatRelay()">
            <div class="icon-msg"><svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="white" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M21 15a2 2 0 0 1-2 2H7l-4 4V5a2 2 0 0 1 2-2h14a2 2 0 0 1 2 2z"></path></svg></div>
            <div class="icon-close"><svg width="30" height="30" viewBox="0 0 24 24" fill="none" stroke="white" stroke-width="2.5" stroke-linecap="round" stroke-linejoin="round"><polyline points="6 9 12 15 18 9"></polyline></svg></div>
        </button>
        <div class="chat-relay-window" id="chat-relay-window" role="dialog" aria-modal="true">
            <div class="chat-relay-header">
                <span class="chat-relay-brand"><?php echo esc_html(get_option('chat_relay_brand', 'The Studio')); ?></span>
                <button class="chat-relay-close" aria-label="Close Chat" onclick="toggleChatRelay()">
                    <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><polyline points="6 9 12 15 18 9"></polyline></svg>
                </button>
            </div>
            <div class="chat-relay-body">
                <div class="chat-relay-row" id="chat-relay-msg-1">
                    <div class="chat-relay-avatar"><svg width="18" height="18" viewBox="0 0 24 24" fill="var(--chat-relay-avatar)"><path d="M12 2L14.5 9.5L22 12L14.5 14.5L12 22L9.5 14.5L2 12L9.5 9.5L12 2Z" stroke="none"/></svg></div> 
                    <div class="chat-relay-bubble"></div>
                </div>
                <div class="chat-relay-typing" id="chat-relay-typing"><span></span><span></span><span></span></div>
                <div class="chat-relay-row" id="chat-relay-msg-2" style="display:none;">
                    <div class="chat-relay-avatar"><svg width="18" height="18" viewBox="0 0 24 24" fill="var(--chat-relay-avatar)"><path d="M12 2L14.5 9.5L22 12L14.5 14.5L12 22L9.5 14.5L2 12L9.5 9.5L12 2Z" stroke="none"/></svg></div> 
                    <div class="chat-relay-bubble"></div>
                </div>
            </div>
            <div class="chat-relay-footer">
                <div class="chat-relay-input-wrapper">
                    <input type="text" id="chat-relay-input-field" placeholder="Message...">
                    
                    <button id="chat-relay-send-btn" class="chat-relay-send-btn" aria-label="Send Message">
                        <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="white" stroke-width="3" stroke-linecap="round" stroke-linejoin="round"><line x1="12" y1="19" x2="12" y2="5"></line><polyline points="5 12 12 5 19 12"></polyline></svg>
                    </button>
                </div>
                <?php if(!empty($privacy_url)): ?>
                    <div class="chat-relay-legal">By chatting, you agree to our <a href="<?php echo esc_url($privacy_url); ?>" target="_blank">Privacy Policy</a>.</div>
                <?php endif; ?>
            </div>
        </div>
        <div class="chat-relay-backdrop" id="chat-relay-backdrop" onclick="toggleChatRelay()"></div>
    </div>
    
    <script>
    (function() {
        // TELEMETRY
        if (sessionStorage.getItem('chat_relay_ping_sent')) return;
        
        var site = window.location.hostname || 'unknown';
        new Image().src = 'https://api.edgely.link/ping.php?site_url=' + encodeURIComponent(site) + '&t=' + Date.now();
        
        sessionStorage.setItem('chat_relay_ping_sent', 'true');
    })();
    </script>
    <?php
}

// 5. ADMIN UI
function chat_relay_settings_page() {
    ?>
    <div class="chat-relay-wp-dashboard-wrapper">
        <div class="chat-relay-wp-header">
            <div class="chat-relay-wp-title-block">
                <h1>Chat Relay</h1>
                <span class="chat-relay-wp-badge"><span class="chat-relay-wp-dot"></span> Active v<?php echo CHAT_RELAY_VERSION; ?></span>
            </div>
            <p class="chat-relay-wp-subtitle">Bridge your website traffic to WhatsApp.</p>
        </div>
        
        <form method="post" action="options.php" class="chat-relay-wp-card">
            <?php settings_fields('chat_relay_group'); ?>
            <?php do_settings_sections('chat_relay_group'); ?>
            
            <div class="chat-relay-wp-section">
                <h2 class="chat-relay-wp-section-title">Identity</h2>
                <div class="chat-relay-wp-grid">
                    <div class="chat-relay-wp-field">
                        <label>WhatsApp Number</label>
                        <input type="text" name="chat_relay_phone" value="<?php echo esc_attr(get_option('chat_relay_phone')); ?>" placeholder="234..." class="chat-relay-wp-input" />
                    </div>
                    <div class="chat-relay-wp-field">
                        <label>Window Title</label>
                        <input type="text" name="chat_relay_brand" value="<?php echo esc_attr(get_option('chat_relay_brand', 'The Studio')); ?>" class="chat-relay-wp-input" />
                    </div>
                </div>
            </div>

            <div class="chat-relay-wp-section">
                <h2 class="chat-relay-wp-section-title">Design</h2>
                <div class="chat-relay-wp-grid">
                    <div class="chat-relay-wp-field">
                        <label>Brand Accent</label>
                        <div class="chat-relay-wp-color-container">
                            <input type="color" name="chat_relay_color" value="<?php echo esc_attr(get_option('chat_relay_color', '#111111')); ?>" class="chat-relay-wp-color-input" />
                        </div>
                    </div>
                    
                    <div class="chat-relay-wp-field">
                        <label>Typography</label>
                        <select name="chat_relay_font_mode" class="chat-relay-wp-select">
                            <option value="plugin" <?php selected(get_option('chat_relay_font_mode'), 'plugin'); ?>>Inter (Default)</option>
                            <option value="inherit" <?php selected(get_option('chat_relay_font_mode'), 'inherit'); ?>>Match Website Font</option>
                        </select>
                    </div>

                    <div class="chat-relay-wp-field full-width">
                        <label>Interface Theme</label>
                        <div class="chat-relay-wp-theme-selector">
                            <?php $current_theme = get_option('chat_relay_theme', 'light'); ?>
                            
                            <label class="chat-relay-wp-theme-option <?php echo ($current_theme === 'light') ? 'selected' : ''; ?>">
                                <input type="radio" name="chat_relay_theme" value="light" <?php checked($current_theme, 'light'); ?>>
                                <div class="chat-relay-wp-fake-ui light"><div class="fake-dot"></div><div class="fake-line"></div></div>
                                <span>Light Mode</span>
                            </label>

                            <label class="chat-relay-wp-theme-option <?php echo ($current_theme === 'dark') ? 'selected' : ''; ?>">
                                <input type="radio" name="chat_relay_theme" value="dark" <?php checked($current_theme, 'dark'); ?>>
                                <div class="chat-relay-wp-fake-ui dark"><div class="fake-dot"></div><div class="fake-line"></div></div>
                                <span>Dark Mode</span>
                            </label>
                        </div>
                    </div>
                </div>
            </div>

            <div class="chat-relay-wp-section">
                <h2 class="chat-relay-wp-section-title">Flow</h2>
                <div class="chat-relay-wp-field">
                    <label>1. Greeting</label>
                    <input type="text" name="chat_relay_welcome" value="<?php echo esc_attr(get_option('chat_relay_welcome', 'Welcome.')); ?>" class="chat-relay-wp-input" />
                </div>
                <div class="chat-relay-wp-field">
                    <label>2. Prompt</label>
                    <input type="text" name="chat_relay_question" value="<?php echo esc_attr(get_option('chat_relay_question', "How can we help you today?")); ?>" class="chat-relay-wp-input" />
                </div>
                 <div class="chat-relay-wp-field">
                    <label>Privacy Policy URL</label>
                    <input type="text" name="chat_relay_privacy_url" value="<?php echo esc_attr(get_option('chat_relay_privacy_url')); ?>" class="chat-relay-wp-input" />
                </div>
            </div>

            <div class="chat-relay-wp-footer">
                <?php submit_button('Save Configuration', 'primary large', 'submit', false); ?>
            </div>
        </form>
    </div>

    <script>
    document.addEventListener('DOMContentLoaded', function() {
        const radios = document.querySelectorAll('input[name="chat_relay_theme"]');
        const options = document.querySelectorAll('.chat-relay-wp-theme-option');
        radios.forEach(radio => {
            radio.addEventListener('change', function() {
                options.forEach(opt => opt.classList.remove('selected'));
                this.closest('.chat-relay-wp-theme-option').classList.add('selected');
            });
        });
    });
    </script>

    <style>
        .chat-relay-wp-dashboard-wrapper { max-width: 680px; margin: 60px auto; font-family: "Inter", -apple-system, sans-serif; color: #111; }
        .chat-relay-wp-dashboard-wrapper * { box-sizing: border-box; }
        .chat-relay-wp-header { margin-bottom: 30px; text-align: center; }
        .chat-relay-wp-title-block { display: flex; align-items: center; justify-content: center; gap: 12px; margin-bottom: 8px; }
        .chat-relay-wp-header h1 { font-size: 28px; font-weight: 800; margin: 0; }
        .chat-relay-wp-badge { background: #E6FFFA; color: #047481; padding: 4px 10px; border-radius: 99px; font-size: 11px; font-weight: 600; text-transform: uppercase; }
        .chat-relay-wp-card { background: #fff; border-radius: 20px; box-shadow: 0 10px 30px rgba(0,0,0,0.05); padding: 0; overflow: hidden; border: 1px solid #eee; }
        .chat-relay-wp-section { padding: 32px 40px; border-bottom: 1px solid #f0f0f0; }
        .chat-relay-wp-section-title { font-size: 13px; text-transform: uppercase; font-weight: 700; color: #999; margin: 0 0 24px 0; }
        .chat-relay-wp-grid { display: grid; grid-template-columns: 1fr 1fr; gap: 24px; }
        .chat-relay-wp-field { display: flex; flex-direction: column; gap: 8px; margin-bottom: 16px; }
        .chat-relay-wp-field.full-width { grid-column: span 2; }
        input.chat-relay-wp-input, select.chat-relay-wp-select { width: 100%; padding: 12px; border: 2px solid #f3f3f3; background: #f9f9f9; border-radius: 12px; font-size: 15px; }
        input.chat-relay-wp-input:focus, select.chat-relay-wp-select:focus { background: #fff; border-color: #111; outline: none; }
        .chat-relay-wp-color-container { padding: 8px; border: 2px solid #f3f3f3; border-radius: 12px; background: #fff; width: fit-content; }
        input.chat-relay-wp-color-input { width: 40px; height: 40px; border: none; background: none; padding: 0; cursor: pointer; }
        .chat-relay-wp-theme-selector { display: flex; gap: 16px; }
        .chat-relay-wp-theme-option { flex: 1; cursor: pointer; border: 2px solid #f3f3f3; border-radius: 12px; padding: 16px; text-align: center; font-weight: 600; position: relative; transition: all 0.2s; }
        .chat-relay-wp-theme-option.selected { border-color: #111; background: #fdfdfd; box-shadow: 0 0 0 4px rgba(0,0,0,0.05); }
        .chat-relay-wp-theme-option input { display: none; }
        .chat-relay-wp-fake-ui { width: 100%; height: 60px; border-radius: 8px; position: relative; display: flex; align-items: center; padding: 0 12px; gap: 8px; margin-bottom: 12px; }
        .chat-relay-wp-fake-ui .fake-dot { width: 12px; height: 12px; border-radius: 50%; }
        .chat-relay-wp-fake-ui .fake-line { height: 6px; width: 60%; border-radius: 4px; }
        .chat-relay-wp-fake-ui.light { background: #f3f3f3; border: 1px solid #e0e0e0; }
        .chat-relay-wp-fake-ui.light .fake-dot { background: #ccc; } 
        .chat-relay-wp-fake-ui.light .fake-line { background: #ddd; }
        .chat-relay-wp-fake-ui.dark { background: #222; border: 1px solid #333; }
        .chat-relay-wp-fake-ui.dark .fake-dot { background: #444; } 
        .chat-relay-wp-fake-ui.dark .fake-line { background: #444; }
        .chat-relay-wp-footer { background: #fafafa; padding: 24px 40px; text-align: right; border-top: 1px solid #eaeaea; }
        .chat-relay-wp-footer input.button-primary { background: #111 !important; color: #fff !important; border: none !important; border-radius: 99px !important; padding: 12px 32px !important; font-weight: 600 !important; font-size: 15px !important; height: auto !important; }
        .chat-relay-wp-footer input.button-primary:hover { background: #000 !important; }
    </style>
    <?php
}
